'use client';

import PageHeader from "@/components/layout/Pages/PageHeader";
import { useEffect, useState, useRef } from "react";
import { useLocale } from 'next-intl';

// 1. Defined Interfaces for type safety
interface LocalizedText {
  ar: string;
  en: string;
}

interface ArticleItem {
  id: number;
  publisher_date: string;
  title: LocalizedText;
  description: LocalizedText;
  image: string;
  tags: string[];
}

export default function Content() {
  const locale = useLocale() as 'ar' | 'en';
  
  // 2. Initialized states with proper types
  const [title, setTitle] = useState<LocalizedText>({ ar: "", en: "" });
  const [description, setDescription] = useState<LocalizedText>({ ar: "", en: "" });
  const [selectedArticle, setSelectedArticle] = useState<ArticleItem | null>(null);
  
  // 3. Fixed Ref type error
  const detailsRef = useRef<HTMLDivElement>(null);

  const handleArticleClick = (item: ArticleItem) => {
    setSelectedArticle(item);
    setTimeout(() => {
      if (detailsRef.current) {
        detailsRef.current.scrollIntoView({
          behavior: "smooth",
          block: "start"
        });
      }
    }, 100);
  };

  useEffect(() => {
    setTitle({
      ar: "أخبار ومقالات ..",
      en: "News & Articles .."
    });
    setDescription({
      ar: "كن على اطلاع دائم على أحدث أخبار شركة السلام، وتقارير المشاريع، والمقالات المتخصصة في قطاع الإنشاء والبناء.",
      en: "Stay up to date with the latest news from Al-Salam Company, project reports, and specialized articles in the construction sector."
    });
  }, []);

  // 4. Example items array following the schema
  const items: ArticleItem[] = [
    {
      id: 1,
      publisher_date: "12 Feb 2026",
      title: { ar: "بدء تنفيذ مشروع العاصمة", en: "Start of Capital Project" },
      description: { ar: "تفاصيل المشروع الجديد...", en: "New project details..." },
      image: "/assets/images/news-1.jpg",
      tags: ["Construction", "2026"]
    }
  ];

  return (
    <>
      {/* 5. Safe access to localized title and description */}
      <PageHeader 
        title={title[locale] || title.ar} 
        description={description[locale] || description.ar} 
      />

      <div className='h-auto w-full flex flex-col justify-start items-center gap-0 mb-20'>
        <div id="newsContent" ref={detailsRef} className="container flex flex-col justify-center items-center gap-5 py-10 px-4 md:px-0">
          
          {/* Search Bar Section */}
          <div className="flex w-full rounded-lg bg-gray-100 h-16 flex-row justify-start items-center px-4">
            <svg className="h-6 w-6" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M23.707 22.2928L17.738 16.3238C19.3647 14.3344 20.1644 11.7959 19.9719 9.23334C19.7793 6.67081 18.6092 4.28031 16.7036 2.5563C14.798 0.832293 12.3026 -0.0933258 9.73367 -0.029094C17.9976 12.1208 17.154 14.1542 15.6543 15.654C14.1545 17.1538 12.121 17.9974 10 17.9998Z" fill="#FF6400" />
            </svg>
            <input 
              placeholder={locale === "ar" ? "ابحث عن مقال..." : "Search for a post..."} 
              className="w-full bg-transparent placeholder:text-gray-400 flex h-full p-4 outline-none text-main-blue" 
            />
          </div>

          {/* Results Info */}
          <div className="flex w-full flex-row justify-start items-center gap-2 px-2 text-sm">
            <span className="text-gray-400 font-almarai font-light">
              {locale === "ar" ? "نتائج البحث" : "Search Results"}
            </span>
            <span className="text-main-orange font-almarai font-bold">
              {items.length} {locale === "ar" ? "مقال" : "Articles"}
            </span>
          </div>

          {/* Articles Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 w-full gap-10 mt-6">
            {items.map((item) => (
              <div 
                key={item.id} 
                onClick={() => handleArticleClick(item)}
                className="cursor-pointer group flex flex-col gap-4 border border-gray-100 rounded-2xl overflow-hidden hover:shadow-lg transition-shadow"
              >
                <div className="relative h-48 w-full overflow-hidden">
                   <img src={item.image} alt={item.title[locale]} className="object-cover w-full h-full group-hover:scale-105 transition-transform duration-500" />
                </div>
                <div className="p-4 flex flex-col gap-2">
                  <span className="text-xs text-main-orange">{item.publisher_date}</span>
                  <h3 className="font-bold text-main-blue text-lg line-clamp-2">{item.title[locale]}</h3>
                  <p className="text-gray-500 text-sm line-clamp-3">{item.description[locale]}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </>
  );
}